//: ## ![3DaysOfSwift.com Logo](3DaysIcon46.png) Unwrapping Optionals
//:
//: There are five options to unwrap an optional:
//: 1. Optional Binding *(✅ safe)*
//: 2. Nil-Coalescing Operator *(✅ safe)*
//: 3. Force Unwrapping Optionals *(⚠️ dangerous)*
//: 4. Implicitly Unwrapping Optionals *(⚠️ dangerous)*
//: 5. Optional Chaining *(✅ safe)*
//:
//: -------------------
//:
//: [◀ Previous Page](@previous) | [Next Page  ▶](@next)
//:
//: -------------------
//:


// custom type using a struct
struct Profile {
    
    // stored properties
    let firstName: String
    let lastName: String
    let nickname: String?
    let title: String! // 4. Implicitly Unwrapping Optionals (⚠️ dangerous)
    let age: Int
    
    // computed properties
    var fullName: String {
        return firstName + " " + lastName
    }
    var displayName: String {
        nickname ?? fullName // 2. Nil-Coalescing Operator (✅ safe)
    }
    
    func getNickNameOrFirstName() -> String {
        if let unwrappedValue = nickname { // 1. Optional Binding (✅ safe)
            return unwrappedValue
        } else {
            return firstName
        }
    }
}

// instantiate an instance of our custom struct
let profile = Profile(firstName: "John", lastName: "Appleseed", nickname: "Johnny Boy", title: "Mr", age: 38)

// use dot notation to access the display name computed property
print("Welcome, \(profile.displayName)")


// 3. Force Unwrapping Optionals (⚠️ dangerous)
// code will crash when no nickname exists
let nickname: String = profile.nickname!
print(nickname)


// 4. Implicitly Unwrapping Optionals (⚠️ dangerous)
// when the stored property is called the value will be force unwrapped and crash when no title exists
let title: String = profile.title
print(title)


// 5. Optional Chaining (✅ safe)
let firstCharacter: String? = profile.nickname?.lowercased().first?.uppercased()
if let unwrappedValue = firstCharacter {
    print(unwrappedValue)
}


//:
//: -------------------
//:
//: ## Task:
//:
//: In the code area below instantiate a new instance of the `Profile` struct, providing `nil` for each optional parameter in the initialiser function.
//:
//: Trigger a runtime error (crash the application) by using the two dangerous options for unwrapping optionals.
//:
//: -------------------
//:
//: [◀ Previous Page](@previous) | [Next Page  ▶](@next)
//:
//: -------------------
//:


// write code here










